﻿// Copyright Epic Games, Inc. All Rights Reserved.

using System.IO;
using UnrealBuildTool;

public class RuntimeAudioImporterSileroVAD : ModuleRules
{
	// In earlier UE versions (such as 4.27), the architecture string is empty.
	protected string ArchitectureString
	{
		get
		{
			if (Target.Architecture.ToString().Length > 0)
			{
				return Target.Architecture.ToString();
			}
			else
			{
				return "x64";
			}
		}
	}
	
	protected void LoadOnnxRuntime()
	{
		string PlatformDir = Target.Platform.ToString();
		string IncDirPath = Path.Combine(ModuleDirectory, "..", "ThirdParty", "Include", "onnxruntime");
		string LibDirPath = "";
		string RelativeLibDirPath = "";
		string SharedLibFileName = "UNSUPPORTED_PLATFORM";

		if (Target.Platform == UnrealTargetPlatform.Win64)
		{
			if (Target.Version.MajorVersion >= 5 && Target.Version.MinorVersion >= 6)
			{
				PublicDependencyModuleNames.Add("NNERuntimeORT");
				PublicDependencyModuleNames.Add("NNEOnnxruntime");
			}
			else
			{
				RelativeLibDirPath = Path.Combine("..", "ThirdParty", "Lib", PlatformDir, "onnxruntime", ArchitectureString);
				LibDirPath = Path.Combine(ModuleDirectory, RelativeLibDirPath);
				SharedLibFileName = "onnxruntime.dll";
				PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, "onnxruntime.lib"));
				RuntimeDependencies.Add(Path.Combine(LibDirPath, SharedLibFileName));
				PublicDelayLoadDLLs.Add(SharedLibFileName);
				PublicIncludePaths.Add(IncDirPath);
			}
		}
		else if (Target.Platform == UnrealTargetPlatform.Linux)
		{
			RelativeLibDirPath = Path.Combine("..", "ThirdParty", "Lib", PlatformDir, "onnxruntime", ArchitectureString);
			LibDirPath = Path.Combine(ModuleDirectory, RelativeLibDirPath);
			SharedLibFileName = "libonnxruntime.so";
			PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, SharedLibFileName));
			RuntimeDependencies.Add(Path.Combine(LibDirPath, "libonnxruntime.so.1"));
			RuntimeDependencies.Add(Path.Combine(LibDirPath, SharedLibFileName));
			PublicIncludePaths.Add(IncDirPath);
		}
		else if (Target.Platform == UnrealTargetPlatform.Mac)
		{
			RelativeLibDirPath = Path.Combine("..", "ThirdParty", "Lib", PlatformDir, "onnxruntime");
			LibDirPath = Path.Combine(ModuleDirectory, RelativeLibDirPath);
			SharedLibFileName = "libonnxruntime.dylib";
			PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, SharedLibFileName));
			RuntimeDependencies.Add(Path.Combine(LibDirPath, "libonnxruntime.1.19.2.dylib"));
			RuntimeDependencies.Add(Path.Combine(LibDirPath, SharedLibFileName));
			PublicIncludePaths.Add(IncDirPath);
		}
		else if (Target.Platform == UnrealTargetPlatform.Android)
		{
			RelativeLibDirPath = Path.Combine("..", "ThirdParty", "Lib", PlatformDir, "onnxruntime");
			LibDirPath = Path.Combine(ModuleDirectory, RelativeLibDirPath);
			SharedLibFileName = "libonnxruntime.so";
			PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, "armeabi-v7a", SharedLibFileName));
			PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, "arm64-v8a", SharedLibFileName));
			PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, "x86", SharedLibFileName));
			PublicAdditionalLibraries.Add(Path.Combine(LibDirPath, "x86_64", SharedLibFileName));

			string RuntimeAudioImporterSileroVAD_APLPath =
				Utils.MakePathRelativeTo(Path.Combine(ModuleDirectory, "RuntimeAudioImporterSileroVAD_APL.xml"),
					Target.RelativeEnginePath);
			AdditionalPropertiesForReceipt.Add("AndroidPlugin", RuntimeAudioImporterSileroVAD_APLPath);
			PublicIncludePaths.Add(IncDirPath);
		}
		else if (Target.Platform == UnrealTargetPlatform.IOS)
		{
			RelativeLibDirPath = Path.Combine("..", "ThirdParty", "Lib", PlatformDir);
			LibDirPath = Path.Combine(ModuleDirectory, RelativeLibDirPath);
			PublicAdditionalFrameworks.Add(
				new Framework(
					"onnxruntime",
					"../ThirdParty/Lib/IOS/onnxruntime.embeddedframework.zip"
				)
			);
			PublicIncludePaths.Add(IncDirPath);
		}

		string RelativeSharedLibPath = Path.Combine(RelativeLibDirPath, SharedLibFileName).Replace('\\', '/');
		PublicDefinitions.Add("ORT_API_MANUAL_INIT");
		if (Target.Type != TargetType.Editor)
		{
			PublicDefinitions.Add("ORT_NO_EXCEPTIONS");
		}

		PublicDefinitions.Add("ONNXRUNTIME_SHAREDLIB_PATH_SILERO_VAD=" + RelativeSharedLibPath);
	}
	
	public RuntimeAudioImporterSileroVAD(ReadOnlyTargetRules Target) : base(Target)
	{
		PCHUsage = ModuleRules.PCHUsageMode.UseExplicitOrSharedPCHs;

		PublicDependencyModuleNames.AddRange(
			new string[]
			{
				"Core",
				"RuntimeAudioImporter"
			}
		);

		PrivateDependencyModuleNames.AddRange(
			new string[]
			{
				"CoreUObject",
				"Engine",
			}
		);
		
		PrivateDependencyModuleNames.Add("Projects");

		LoadOnnxRuntime();
	}
}