﻿// Georgy Treshchev 2025.

#pragma once

#include "CoreMinimal.h"
#include "Sound/CapturableSoundWave.h"
#include "IPixelStreaming2AudioConsumer.h"
#include "IPixelStreaming2AudioSink.h"
#include <atomic>
#include "Tickable.h"
#include "Misc/EngineVersionComparison.h"
#include "PixelStreaming2CapturableSoundWave.generated.h"

/**
 * Information about a Pixel Streaming player that can be used as an audio source
 */
USTRUCT(BlueprintType, Category = "Pixel Streaming Capturable Sound Wave")
struct RUNTIMEAUDIOIMPORTERPIXELSTREAMING2_API FPixelStreaming2PlayerInfo_RAI
{
	GENERATED_BODY()

	/** The ID of the player */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Pixel Streaming Player Info")
	FString PlayerId;

	/** The ID of the streamer this player is connected to */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Pixel Streaming Player Info")
	FString StreamerId;

	/** Default constructor */
	FPixelStreaming2PlayerInfo_RAI()
	{
	}

	/** Constructor with parameters */
	FPixelStreaming2PlayerInfo_RAI(const FString& InPlayerId, const FString& InStreamerId)
		: PlayerId(InPlayerId)
		, StreamerId(InStreamerId)
	{
	}
};

/** Static delegate broadcasting available Pixel Streaming players */
DECLARE_DELEGATE_OneParam(FOnGetAvailablePixelStreaming2PlayersResultNative, const TArray<FPixelStreaming2PlayerInfo_RAI>&);

/** Dynamic delegate broadcasting available Pixel Streaming players */
DECLARE_DYNAMIC_DELEGATE_OneParam(FOnGetAvailablePixelStreaming2PlayersResult, const TArray<FPixelStreaming2PlayerInfo_RAI>&, AvailablePlayers);

/**
 * Sound wave that can capture audio data from Pixel Streaming input devices (e.g., browser microphone)
 */
UCLASS(BlueprintType, Category = "Pixel Streaming Capturable Sound Wave")
class RUNTIMEAUDIOIMPORTERPIXELSTREAMING2_API UPixelStreaming2CapturableSoundWave : public UCapturableSoundWave, public IPixelStreaming2AudioConsumer, public FTickableGameObject
{
	GENERATED_BODY()

public:
	UPixelStreaming2CapturableSoundWave(const FObjectInitializer& ObjectInitializer);

	/**
	 * Create a new instance of the pixel streaming capturable sound wave.
	 *
	 * @return A new instance of UPixelStreaming2CapturableSoundWave.
	 */
	UFUNCTION(BlueprintCallable, Category = "Pixel Streaming Capturable Sound Wave|Main")
	static UPixelStreaming2CapturableSoundWave* CreatePixelStreaming2CapturableSoundWave();

	/**
	 * Get information about all available Pixel Streaming players
	 * 
	 * @param Result Delegate broadcasting the result
	 */
	UFUNCTION(BlueprintCallable, Category = "Pixel Streaming Capturable Sound Wave|Info")
	static void GetAvailablePixelStreaming2Players(const FOnGetAvailablePixelStreaming2PlayersResult& Result);

	/**
	 * Gets information about all Pixel Streaming players available in the system. Suitable for use in C++
	 * 
	 * @param Result Delegate broadcasting the result
	 */
	static void GetAvailablePixelStreaming2Players(const FOnGetAvailablePixelStreaming2PlayersResultNative& Result);

	//~ Begin UCapturableSoundWave Interface
	virtual bool StartCapture_Implementation(int32 DeviceId) override;
	virtual void StopCapture_Implementation() override;
	virtual bool ToggleMute_Implementation(bool bMute) override;
	virtual bool IsCapturing_Implementation() const override;
	//~ End UCapturableSoundWave Interface

	//~ Begin UObject Interface
	virtual void BeginDestroy() override;
	//~ End UObject Interface

	//~ Begin FTickableGameObject Interface
	virtual void Tick(float DeltaTime) override;
	virtual bool IsTickable() const override;
	virtual TStatId GetStatId() const override;
	//~ End FTickableGameObject Interface

	//~ Begin IPixelStreamingAudioConsumer Interface
	virtual void ConsumeRawPCM(const int16_t* AudioData, int InSampleRate, size_t NChannels, size_t NFrames) override;

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 7
	virtual void OnAudioConsumerAdded() override;
	virtual void OnAudioConsumerRemoved() override;
#else
	virtual void OnConsumerAdded() override;
	virtual void OnConsumerRemoved() override;
#endif
	//~ End IPixelStreamingAudioConsumer Interface

	/**
	 * The Pixel Streaming player information for the player whose audio we wish to capture.
	 * If the PlayerId is left blank, this component will listen to the first non-listened to peer that connects.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Pixel Streaming Capturable Sound Wave")
	FPixelStreaming2PlayerInfo_RAI PlayerInfo;

private:
	/** Pixel Streaming audio sink that provides the audio data */
	TWeakPtr<IPixelStreaming2AudioSink> AudioSink;

	/** Whether we're currently capturing audio */
	std::atomic<bool> bIsCapturing;

	/** Whether we're currently muted */
	std::atomic<bool> bIsMuted;

	/** Whether we've requested capture but haven't connected yet */
	std::atomic<bool> bCaptureRequested;

	/** Try to find and connect to a Pixel Streaming audio source */
	bool TryConnectToPixelStreaming2Audio();
};