﻿// Georgy Treshchev 2025.

#include "PixelStreaming2CapturableSoundWave.h"
#include "IPixelStreaming2Module.h"
#include "IPixelStreaming2Streamer.h"
#include "RuntimeAudioImporterPixelStreaming2.h"
#include "Misc/EngineVersionComparison.h"

UPixelStreaming2CapturableSoundWave::UPixelStreaming2CapturableSoundWave(const FObjectInitializer& ObjectInitializer)
	: Super(ObjectInitializer)
	, AudioSink(nullptr)
	, bIsCapturing(false)
	, bIsMuted(false)
	, bCaptureRequested(false)
{
}

UPixelStreaming2CapturableSoundWave* UPixelStreaming2CapturableSoundWave::CreatePixelStreaming2CapturableSoundWave()
{
	if (!IsInGameThread())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Error, TEXT("Unable to create a sound wave outside of the game thread"));
		return nullptr;
	}

	return NewObject<UPixelStreaming2CapturableSoundWave>();
}

void UPixelStreaming2CapturableSoundWave::GetAvailablePixelStreaming2Players(const FOnGetAvailablePixelStreaming2PlayersResult& Result)
{
	GetAvailablePixelStreaming2Players(FOnGetAvailablePixelStreaming2PlayersResultNative::CreateLambda([Result](const TArray<FPixelStreaming2PlayerInfo_RAI>& AvailablePlayers)
	{
		Result.ExecuteIfBound(AvailablePlayers);
	}));
}

void UPixelStreaming2CapturableSoundWave::GetAvailablePixelStreaming2Players(const FOnGetAvailablePixelStreaming2PlayersResultNative& Result)
{
#if UE_VERSION_NEWER_THAN(5, 2, 9)
	// Execute on game thread to ensure thread safety
	if (!IsInGameThread())
	{
		AsyncTask(ENamedThreads::GameThread, [Result]()
		{
			GetAvailablePixelStreaming2Players(Result);
		});
		return;
	}

	TArray<FPixelStreaming2PlayerInfo_RAI> AvailablePlayers;

	if (!IPixelStreaming2Module::IsAvailable())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Verbose, TEXT("Cannot get Pixel Streaming players because the Pixel Streaming module is not loaded"));
		Result.ExecuteIfBound(AvailablePlayers);
		return;
	}

	IPixelStreaming2Module& PixelStreamingModule = IPixelStreaming2Module::Get();
	if (!PixelStreamingModule.IsReady())
	{
		Result.ExecuteIfBound(AvailablePlayers);
		return;
	}

	// Get all streamers
	TArray<FString> StreamerIds = PixelStreamingModule.GetStreamerIds();
	
	// If no streamers, try the default one
	if (StreamerIds.Num() == 0)
	{
		FString DefaultStreamerId = PixelStreamingModule.GetDefaultStreamerID();
		if (!DefaultStreamerId.IsEmpty())
		{
			StreamerIds.Add(DefaultStreamerId);
		}
	}

	// For each streamer, get the connected players
	for (const FString& StreamerId : StreamerIds)
	{
		TSharedPtr<IPixelStreaming2Streamer> Streamer = PixelStreamingModule.FindStreamer(StreamerId);
		if (!Streamer)
		{
			continue;
		}

		// Get all connected peers for this streamer
		TArray<FString> ConnectedPeers = Streamer->GetConnectedPlayers();
		
		// Add each player to the result array
		for (const FString& PlayerId : ConnectedPeers)
		{
			UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Found Pixel Streaming player '%s' connected to streamer '%s'"), *PlayerId, *StreamerId);
			AvailablePlayers.Add(FPixelStreaming2PlayerInfo_RAI(PlayerId, StreamerId));
		}
	}

	Result.ExecuteIfBound(AvailablePlayers);
#else
	UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Error, TEXT("Unable to get available Pixel Streaming players as GetAvailablePixelStreaming2Players is not supported in this version of UE (requires 5.3 or newer)"));
#endif
}

bool UPixelStreaming2CapturableSoundWave::StartCapture_Implementation(int32 DeviceId)
{
	// DeviceId is ignored for Pixel Streaming as we use PlayerToHear/StreamerToHear instead
	if (bIsCapturing)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Warning, TEXT("Pixel Streaming capture is already running"));
		return true;
	}

	bIsMuted = false;
	bCaptureRequested = true;
	
	if (TryConnectToPixelStreaming2Audio())
	{
		bIsCapturing = true;
		bCaptureRequested = false;
		return true;
	}
	
	// We'll try to connect in the Tick function
	return true;
}

void UPixelStreaming2CapturableSoundWave::StopCapture_Implementation()
{
	if (!bIsCapturing && !bCaptureRequested)
	{
		return;
	}

	if (TSharedPtr<IPixelStreaming2AudioSink> PinnedSink = AudioSink.Pin())
	{
		PinnedSink->RemoveAudioConsumer(TWeakPtrVariant<IPixelStreaming2AudioConsumer>(this));
		AudioSink = nullptr;
	}

	bIsCapturing = false;
	bIsMuted = false;
	bCaptureRequested = false;
}

bool UPixelStreaming2CapturableSoundWave::ToggleMute_Implementation(bool bMute)
{
	if (bMute == bIsMuted)
	{
		return true;
	}

	bIsMuted = bMute;
	
	if (bMute)
	{
		// Keep the connection but don't process audio
		return true;
	}
	else
	{
		// Resume processing audio
		return true;
	}
}

bool UPixelStreaming2CapturableSoundWave::IsCapturing_Implementation() const
{
	return bIsCapturing && !bIsMuted;
}

void UPixelStreaming2CapturableSoundWave::BeginDestroy()
{
	if ((bIsCapturing || bCaptureRequested) && AudioSink.IsValid())
	{
		AudioSink.Pin()->RemoveAudioConsumer(TWeakPtrVariant<IPixelStreaming2AudioConsumer>(this));
	}
	Super::BeginDestroy();
}

void UPixelStreaming2CapturableSoundWave::Tick(float DeltaTime)
{
	// Only try to connect if we've requested capture but haven't connected yet
	if (bCaptureRequested && !bIsCapturing)
	{
		if (TryConnectToPixelStreaming2Audio())
		{
			bIsCapturing = true;
			bCaptureRequested = false;
		}
	}
}

bool UPixelStreaming2CapturableSoundWave::IsTickable() const
{
	return bCaptureRequested;
}

TStatId UPixelStreaming2CapturableSoundWave::GetStatId() const
{
	RETURN_QUICK_DECLARE_CYCLE_STAT(UPixelStreaming2CapturableSoundWave, STATGROUP_Tickables);
}

void UPixelStreaming2CapturableSoundWave::ConsumeRawPCM(const int16_t* AudioData, int InSampleRate, size_t NChannels, size_t NFrames)
{
	if (!bIsCapturing || bIsMuted)
	{
		return;
	}

	// Convert int16 PCM to float32 PCM
	const size_t NumSamples = NFrames * NChannels;
	TArray<float> FloatData;
	FloatData.SetNumUninitialized(NumSamples);

	// Convert int16 to float
	for (size_t i = 0; i < NumSamples; ++i)
	{
		FloatData[i] = AudioData[i] / 32768.0f;
	}

	// Calculate the size in bytes
	const int32 DataSizeInBytes = NumSamples * sizeof(float);

	// Append the audio data
	AppendAudioDataFromRAW(
		TArray<uint8>(reinterpret_cast<const uint8*>(FloatData.GetData()), DataSizeInBytes),
		ERuntimeRAWAudioFormat::Float32,
		InSampleRate,
		NChannels
	);
}

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 7
void UPixelStreaming2CapturableSoundWave::OnAudioConsumerAdded()
#else
void UPixelStreaming2CapturableSoundWave::OnConsumerAdded()
#endif
{
	bIsCapturing = true;
	bCaptureRequested = false;
}

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 7
void UPixelStreaming2CapturableSoundWave::OnAudioConsumerRemoved()
#else
void UPixelStreaming2CapturableSoundWave::OnConsumerRemoved()
#endif
{
	bIsCapturing = false;
	AudioSink = nullptr;
}

bool UPixelStreaming2CapturableSoundWave::TryConnectToPixelStreaming2Audio()
{
	if (!IPixelStreaming2Module::IsAvailable())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming module is not loaded"));
		return false;
	}

	IPixelStreaming2Module& PixelStreamingModule = IPixelStreaming2Module::Get();
	if (!PixelStreamingModule.IsReady())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming module is not ready"));
		return false;
	}

	// Determine which streamer to use
	FString EffectiveStreamerId = PlayerInfo.StreamerId;
	if (EffectiveStreamerId.IsEmpty())
	{
		TArray<FString> StreamerIds = PixelStreamingModule.GetStreamerIds();
		if (StreamerIds.Num() > 0)
		{
			EffectiveStreamerId = StreamerIds[0];
		}
		else
		{
			EffectiveStreamerId = PixelStreamingModule.GetDefaultStreamerID();
		}
	}

	TSharedPtr<IPixelStreaming2Streamer> Streamer = PixelStreamingModule.FindStreamer(EffectiveStreamerId);
	if (!Streamer)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming streamer '%s' is not found"), *EffectiveStreamerId);
		return false;
	}

	// Get the appropriate audio sink
	bool bListenToAnyPlayer = PlayerInfo.PlayerId.IsEmpty();
	TWeakPtr<IPixelStreaming2AudioSink> CandidateSink = bListenToAnyPlayer 
		? Streamer->GetUnlistenedAudioSink() 
		: Streamer->GetPeerAudioSink(PlayerInfo.PlayerId);

	if (CandidateSink == nullptr)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming streamer '%s' does not have a sink for the player '%s'"), *EffectiveStreamerId, *PlayerInfo.PlayerId);
		return false;
	}

	// Clean up previous connection if any
	if (TSharedPtr<IPixelStreaming2AudioSink> PinnedSink = AudioSink.Pin())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Disconnecting Pixel Streaming audio sink from the previous streamer '%s'"), *Streamer->GetId());
		PinnedSink->RemoveAudioConsumer(TWeakPtrVariant<IPixelStreaming2AudioConsumer>(this));
	}

	// Set up new connection
	UE_LOG(LogRuntimeAudioImporterPixelStreaming2, Log, TEXT("Connecting Pixel Streaming audio sink to the streamer '%s'"), *Streamer->GetId());
	AudioSink = CandidateSink;
	AudioSink.Pin()->AddAudioConsumer(TWeakPtrVariant<IPixelStreaming2AudioConsumer>(this));

	return true;
}