﻿// Georgy Treshchev 2025.

#pragma once

#include "CoreMinimal.h"
#include "Sound/CapturableSoundWave.h"
#include "IPixelStreamingAudioConsumer.h"
#include "IPixelStreamingAudioSink.h"
#include "PixelStreamingPlayerId.h"
#include <atomic>
#include "PixelStreamingCapturableSoundWave.generated.h"

/**
 * Information about a Pixel Streaming player that can be used as an audio source
 */
USTRUCT(BlueprintType, Category = "Pixel Streaming Capturable Sound Wave")
struct RUNTIMEAUDIOIMPORTERPIXELSTREAMING_API FPixelStreamingPlayerInfo_RAI
{
	GENERATED_BODY()

	/** The ID of the player */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Pixel Streaming Player Info")
	FString PlayerId;

	/** The ID of the streamer this player is connected to */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Pixel Streaming Player Info")
	FString StreamerId;

	/** Default constructor */
	FPixelStreamingPlayerInfo_RAI()
	{
	}

	/** Constructor with parameters */
	FPixelStreamingPlayerInfo_RAI(const FString& InPlayerId, const FString& InStreamerId)
		: PlayerId(InPlayerId)
		, StreamerId(InStreamerId)
	{
	}
};

/** Static delegate broadcasting available Pixel Streaming players */
DECLARE_DELEGATE_OneParam(FOnGetAvailablePixelStreamingPlayersResultNative, const TArray<FPixelStreamingPlayerInfo_RAI>&);

/** Dynamic delegate broadcasting available Pixel Streaming players */
DECLARE_DYNAMIC_DELEGATE_OneParam(FOnGetAvailablePixelStreamingPlayersResult, const TArray<FPixelStreamingPlayerInfo_RAI>&, AvailablePlayers);

/**
 * Sound wave that can capture audio data from Pixel Streaming input devices (e.g., browser microphone)
 */
UCLASS(BlueprintType, Category = "Pixel Streaming Capturable Sound Wave")
class RUNTIMEAUDIOIMPORTERPIXELSTREAMING_API UPixelStreamingCapturableSoundWave : public UCapturableSoundWave, public IPixelStreamingAudioConsumer, public FTickableGameObject
{
	GENERATED_BODY()

public:
	UPixelStreamingCapturableSoundWave(const FObjectInitializer& ObjectInitializer);

	/**
	 * Create a new instance of the pixel streaming capturable sound wave.
	 *
	 * @return A new instance of UPixelStreamingCapturableSoundWave.
	 */
	UFUNCTION(BlueprintCallable, Category = "Pixel Streaming Capturable Sound Wave|Main")
	static UPixelStreamingCapturableSoundWave* CreatePixelStreamingCapturableSoundWave();

	/**
	 * Get information about all available Pixel Streaming players
	 * 
	 * @param Result Delegate broadcasting the result
	 */
	UFUNCTION(BlueprintCallable, Category = "Pixel Streaming Capturable Sound Wave|Info")
	static void GetAvailablePixelStreamingPlayers(const FOnGetAvailablePixelStreamingPlayersResult& Result);

	/**
	 * Gets information about all Pixel Streaming players available in the system. Suitable for use in C++
	 * 
	 * @param Result Delegate broadcasting the result
	 */
	static void GetAvailablePixelStreamingPlayers(const FOnGetAvailablePixelStreamingPlayersResultNative& Result);

	//~ Begin UCapturableSoundWave Interface
	virtual bool StartCapture_Implementation(int32 DeviceId) override;
	virtual void StopCapture_Implementation() override;
	virtual bool ToggleMute_Implementation(bool bMute) override;
	virtual bool IsCapturing_Implementation() const override;
	//~ End UCapturableSoundWave Interface

	//~ Begin UObject Interface
	virtual void BeginDestroy() override;
	//~ End UObject Interface

	//~ Begin FTickableGameObject Interface
	virtual void Tick(float DeltaTime) override;
	virtual bool IsTickable() const override;
	virtual TStatId GetStatId() const override;
	//~ End FTickableGameObject Interface

	//~ Begin IPixelStreamingAudioConsumer Interface
	virtual void ConsumeRawPCM(const int16_t* AudioData, int InSampleRate, size_t NChannels, size_t NFrames) override;
	virtual void OnConsumerAdded() override;
	virtual void OnConsumerRemoved() override;
	//~ End IPixelStreamingAudioConsumer Interface

	/**
	 * The Pixel Streaming player information for the player whose audio we wish to capture.
	 * If the PlayerId is left blank, this component will listen to the first non-listened to peer that connects.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Pixel Streaming Capturable Sound Wave")
	FPixelStreamingPlayerInfo_RAI PlayerInfo;

private:
	/** Pixel Streaming audio sink that provides the audio data */
	IPixelStreamingAudioSink* AudioSink;

	/** Whether we're currently capturing audio */
	std::atomic<bool> bIsCapturing;

	/** Whether we're currently muted */
	std::atomic<bool> bIsMuted;

	/** Whether we've requested capture but haven't connected yet */
	std::atomic<bool> bCaptureRequested;

	/** Try to find and connect to a Pixel Streaming audio source */
	bool TryConnectToPixelStreamingAudio();
};