﻿// Georgy Treshchev 2025.

#include "PixelStreamingCapturableSoundWave.h"
#include "IPixelStreamingModule.h"
#include "IPixelStreamingStreamer.h"
#include "RuntimeAudioImporterPixelStreaming.h"
#include "Misc/EngineVersionComparison.h"

UPixelStreamingCapturableSoundWave::UPixelStreamingCapturableSoundWave(const FObjectInitializer& ObjectInitializer)
	: Super(ObjectInitializer)
	, AudioSink(nullptr)
	, bIsCapturing(false)
	, bIsMuted(false)
	, bCaptureRequested(false)
{
}

UPixelStreamingCapturableSoundWave* UPixelStreamingCapturableSoundWave::CreatePixelStreamingCapturableSoundWave()
{
	if (!IsInGameThread())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Error, TEXT("Unable to create a sound wave outside of the game thread"));
		return nullptr;
	}

	return NewObject<UPixelStreamingCapturableSoundWave>();
}

void UPixelStreamingCapturableSoundWave::GetAvailablePixelStreamingPlayers(const FOnGetAvailablePixelStreamingPlayersResult& Result)
{
	GetAvailablePixelStreamingPlayers(FOnGetAvailablePixelStreamingPlayersResultNative::CreateLambda([Result](const TArray<FPixelStreamingPlayerInfo_RAI>& AvailablePlayers)
	{
		Result.ExecuteIfBound(AvailablePlayers);
	}));
}

void UPixelStreamingCapturableSoundWave::GetAvailablePixelStreamingPlayers(const FOnGetAvailablePixelStreamingPlayersResultNative& Result)
{
#if UE_VERSION_NEWER_THAN(5, 2, 9)
	// Execute on game thread to ensure thread safety
	if (!IsInGameThread())
	{
		AsyncTask(ENamedThreads::GameThread, [Result]()
		{
			GetAvailablePixelStreamingPlayers(Result);
		});
		return;
	}

	TArray<FPixelStreamingPlayerInfo_RAI> AvailablePlayers;

	if (!IPixelStreamingModule::IsAvailable())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Verbose, TEXT("Cannot get Pixel Streaming players because the Pixel Streaming module is not loaded"));
		Result.ExecuteIfBound(AvailablePlayers);
		return;
	}

	IPixelStreamingModule& PixelStreamingModule = IPixelStreamingModule::Get();
	if (!PixelStreamingModule.IsReady())
	{
		Result.ExecuteIfBound(AvailablePlayers);
		return;
	}

	// Get all streamers
	TArray<FString> StreamerIds = PixelStreamingModule.GetStreamerIds();
	
	// If no streamers, try the default one
	if (StreamerIds.Num() == 0)
	{
		FString DefaultStreamerId = PixelStreamingModule.GetDefaultStreamerID();
		if (!DefaultStreamerId.IsEmpty())
		{
			StreamerIds.Add(DefaultStreamerId);
		}
	}

	// For each streamer, get the connected players
	for (const FString& StreamerId : StreamerIds)
	{
		TSharedPtr<IPixelStreamingStreamer> Streamer = PixelStreamingModule.FindStreamer(StreamerId);
		if (!Streamer)
		{
			continue;
		}

		// Get all connected peers for this streamer
		TArray<FPixelStreamingPlayerId> ConnectedPeers = Streamer->GetConnectedPlayers();
		
		// Add each player to the result array
		for (const FPixelStreamingPlayerId& PlayerId : ConnectedPeers)
		{
			UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Found Pixel Streaming player '%s' connected to streamer '%s'"), *PlayerId, *StreamerId);
			AvailablePlayers.Add(FPixelStreamingPlayerInfo_RAI(PlayerId, StreamerId));
		}
	}

	Result.ExecuteIfBound(AvailablePlayers);
#else
	UE_LOG(LogRuntimeAudioImporterPixelStreaming, Error, TEXT("Unable to get available Pixel Streaming players as GetAvailablePixelStreamingPlayers is not supported in this version of UE (requires 5.3 or newer)"));
#endif
}

bool UPixelStreamingCapturableSoundWave::StartCapture_Implementation(int32 DeviceId)
{
	// DeviceId is ignored for Pixel Streaming as we use PlayerToHear/StreamerToHear instead
	if (bIsCapturing)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Warning, TEXT("Pixel Streaming capture is already running"));
		return true;
	}

	bIsMuted = false;
	bCaptureRequested = true;
	
	if (TryConnectToPixelStreamingAudio())
	{
		bIsCapturing = true;
		bCaptureRequested = false;
		return true;
	}
	
	// We'll try to connect in the Tick function
	return true;
}

void UPixelStreamingCapturableSoundWave::StopCapture_Implementation()
{
	if (!bIsCapturing && !bCaptureRequested)
	{
		return;
	}

	if (AudioSink)
	{
		AudioSink->RemoveAudioConsumer(this);
		AudioSink = nullptr;
	}

	bIsCapturing = false;
	bIsMuted = false;
	bCaptureRequested = false;
}

bool UPixelStreamingCapturableSoundWave::ToggleMute_Implementation(bool bMute)
{
	if (bMute == bIsMuted)
	{
		return true;
	}

	bIsMuted = bMute;
	
	if (bMute)
	{
		// Keep the connection but don't process audio
		return true;
	}
	else
	{
		// Resume processing audio
		return true;
	}
}

bool UPixelStreamingCapturableSoundWave::IsCapturing_Implementation() const
{
	return bIsCapturing && !bIsMuted;
}

void UPixelStreamingCapturableSoundWave::BeginDestroy()
{
	if ((bIsCapturing || bCaptureRequested) && AudioSink)
	{
		AudioSink->RemoveAudioConsumer(this);
	}
	Super::BeginDestroy();
}

void UPixelStreamingCapturableSoundWave::Tick(float DeltaTime)
{
	// Only try to connect if we've requested capture but haven't connected yet
	if (bCaptureRequested && !bIsCapturing)
	{
		if (TryConnectToPixelStreamingAudio())
		{
			bIsCapturing = true;
			bCaptureRequested = false;
		}
	}
}

bool UPixelStreamingCapturableSoundWave::IsTickable() const
{
	return bCaptureRequested;
}

TStatId UPixelStreamingCapturableSoundWave::GetStatId() const
{
	RETURN_QUICK_DECLARE_CYCLE_STAT(UPixelStreamingCapturableSoundWave, STATGROUP_Tickables);
}

void UPixelStreamingCapturableSoundWave::ConsumeRawPCM(const int16_t* AudioData, int InSampleRate, size_t NChannels, size_t NFrames)
{
	if (!bIsCapturing || bIsMuted)
	{
		return;
	}

	// Convert int16 PCM to float32 PCM
	const size_t NumSamples = NFrames * NChannels;
	TArray<float> FloatData;
	FloatData.SetNumUninitialized(NumSamples);

	// Convert int16 to float
	for (size_t i = 0; i < NumSamples; ++i)
	{
		FloatData[i] = AudioData[i] / 32768.0f;
	}

	// Calculate the size in bytes
	const int32 DataSizeInBytes = NumSamples * sizeof(float);

	// Append the audio data
	AppendAudioDataFromRAW(
		TArray<uint8>(reinterpret_cast<const uint8*>(FloatData.GetData()), DataSizeInBytes),
		ERuntimeRAWAudioFormat::Float32,
		InSampleRate,
		NChannels
	);
}

void UPixelStreamingCapturableSoundWave::OnConsumerAdded()
{
	bIsCapturing = true;
	bCaptureRequested = false;
}

void UPixelStreamingCapturableSoundWave::OnConsumerRemoved()
{
	bIsCapturing = false;
	AudioSink = nullptr;
}

bool UPixelStreamingCapturableSoundWave::TryConnectToPixelStreamingAudio()
{
	if (!IPixelStreamingModule::IsAvailable())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming module is not loaded"));
		return false;
	}

	IPixelStreamingModule& PixelStreamingModule = IPixelStreamingModule::Get();
	if (!PixelStreamingModule.IsReady())
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming module is not ready"));
		return false;
	}

	// Determine which streamer to use
	FString EffectiveStreamerId = PlayerInfo.StreamerId;
	if (EffectiveStreamerId.IsEmpty())
	{
		TArray<FString> StreamerIds = PixelStreamingModule.GetStreamerIds();
		if (StreamerIds.Num() > 0)
		{
			EffectiveStreamerId = StreamerIds[0];
		}
		else
		{
			EffectiveStreamerId = PixelStreamingModule.GetDefaultStreamerID();
		}
	}

	TSharedPtr<IPixelStreamingStreamer> Streamer = PixelStreamingModule.FindStreamer(EffectiveStreamerId);
	if (!Streamer)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming streamer '%s' is not found"), *EffectiveStreamerId);
		return false;
	}

	// Get the appropriate audio sink
	bool bListenToAnyPlayer = PlayerInfo.PlayerId.IsEmpty();
	IPixelStreamingAudioSink* CandidateSink = bListenToAnyPlayer 
		? Streamer->GetUnlistenedAudioSink() 
		: Streamer->GetPeerAudioSink(FPixelStreamingPlayerId(PlayerInfo.PlayerId));

	if (CandidateSink == nullptr)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Cannot capture Pixel Streaming audio because the Pixel Streaming streamer '%s' does not have a sink for the player '%s'"), *EffectiveStreamerId, *PlayerInfo.PlayerId);
		return false;
	}

	// Clean up previous connection if any
	if (AudioSink)
	{
		UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Disconnecting Pixel Streaming audio sink from the previous streamer '%s'"), *Streamer->GetId());
		AudioSink->RemoveAudioConsumer(this);
	}

	// Set up new connection
	UE_LOG(LogRuntimeAudioImporterPixelStreaming, Log, TEXT("Connecting Pixel Streaming audio sink to the streamer '%s'"), *Streamer->GetId());
	AudioSink = CandidateSink;
	AudioSink->AddAudioConsumer(this);

	return true;
}